
#ifndef CE2_CMLA_H
#define CE2_CMLA_H

/*
* All the includes that are needed for code using this module to
* Compile correctly should be #included here.
*/
 
#include "CE2_public.h"

/*
 *  Object %CE2_CMLA.h    : %
 *  State           :  %state%
 *  Creation date   :  Wed Aug 24 2006
 *  Last modified   :  %modify_time%
 */
/** @file
 *  \brief This module contains the prototypes of routines performing SMLA schemes operations.
 *
 *   Implemented according to "CMLA Technical Specification". Version: v1.0-05-12-21,
 *   This material also contains confidential information which may not be disclosed
 *   to others without the prior written consent of CMLA, LLC.
 *
 *  \version CE2_CMLA.h#1:cincl:6
 *  \author  R.Levin.
 *  \remarks Copyright (C) 2006 by Discretix Technologies Ltd.
 *           All Rights reserved
 */

#ifdef __cplusplus
extern "C"
{
#endif

/************************ Defines ******************************/
/************************ Enums ********************************/
/************************ Typedefs  ****************************/

/* CE2_CMLA common types definitions */
#define CE2_CMLA_KEK_SIZE_IN_BYTES       16
typedef DxUint8_t   CE2_CMLA_KEK_t[CE2_CMLA_KEK_SIZE_IN_BYTES];

/* CE2_CMLA_KDF types definitions */
#define CE2_CMLA_SEED_SIZE_IN_BYTES     128
typedef DxUint8_t   CE2_CMLA_SEED_t[CE2_CMLA_SEED_SIZE_IN_BYTES];

/* CE2_CMLA_WRAP types definitions */
#define CE2_CMLA_UNWRAPPED_KEY_SIZE_IN_BYTES   32
#define CE2_CMLA_WRAPPED_KEY_SIZE_IN_BYTES     40
typedef DxUint8_t CE2_CMLA_UNWRAPPED_KEY_t[CE2_CMLA_UNWRAPPED_KEY_SIZE_IN_BYTES];
typedef DxUint8_t CE2_CMLA_WRAPPED_KEY_t[CE2_CMLA_WRAPPED_KEY_SIZE_IN_BYTES];

/* CE2_CMLA_RSA types definitions */
#define CE2_CMLA_RSA_MESSAGE_LEN_BYTES  128
typedef DxUint8_t CE2_CMLA_RSA_MESSAGE_t[CE2_CMLA_RSA_MESSAGE_LEN_BYTES];


/************************ Structs  ******************************/
/************************ Public Variables **********************/
/************************ Public Functions **********************/

/**************************************************************************
 *	              CE2_CMLA_KDF function                                  *
 **************************************************************************/
/**
   @brief  CMLA Key Derivation Function produce the Key Encrypting Key
           [CMLA v1.0-051221].

   CE2_CMLA_KDF operates on an octet string of length 128.
   The input splits in two blocks of equal length. A constant value is concatenated
   to the first block. Then SHA-1 is applied to the preceding result. The result
   and two integers derived from the second block are used in a modulus calculation
   to produce the Key Encrypting Key. The output shall be 16 bytes.


   @param[in]  X        - Seed value, an octet string of length 128.
   @param[out] KEK      - Key Encryption Key, an octet string of length 16.

   @return CE2Error_t - CE2_OK, or error message
                         CE2_CMLA_IS_NOT_SUPPORTED
                         CE2_CMLA_KDF_INVALID_SEED_POINTER_ERROR
                         CE2_CMLA_KDF_INVALID_KEK_POINTER_ERROR

*/
CE2CIMPORT_C  CE2Error_t CE2_CMLA_KDF(CE2_CMLA_SEED_t X,    /*in*/
                                   CE2_CMLA_KEK_t  KEK); /*out*/



/**************************************************************************
 *	              CE2_CMLA_Wrap function                                 *
 **************************************************************************/
/**
   @brief  The CE2_CMLA_Wrap primitive composing CE2 AES Key Wrap Algorithm
           and CMLA_DDT_Perm according to CMLA algorithm [CMLA v1.0-051221].


   @param[in]  KEK      - Key Encryption Key, an octet string of length 16 bytes.
   @param[in]  Key      - Plain key, an octet string of length 32 bytes.
   @param[out] WrapKey  - Wrapped Key, an octet string of length 40 bytes.

   @return CE2Error_t - CE2_OK, or error message
                         CE2_CMLA_IS_NOT_SUPPORTED
                         CE2_CMLA_WRAP_ILLEGAL_KEK_PTR_ERROR
                         CE2_CMLA_WRAP_ILLEGAL_KEY_PTR_ERROR
                         CE2_CMLA_WRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR

*/
CE2CIMPORT_C  CE2Error_t CE2_CMLA_Wrap(
                            CE2_CMLA_KEK_t            KEK,    /*in*/
                            CE2_CMLA_UNWRAPPED_KEY_t  Key,    /*in*/
                            CE2_CMLA_WRAPPED_KEY_t    WrapKey /*out*/ );


/**************************************************************************
 *	              CE2_CMLA_Unwrap function                               *
 **************************************************************************/
/**
   @brief  The CE2_CMLA_Unwrap primitive performs inverse CE2_CMLA_Wrap transformation
           [CMLA v1.0-051221].

   @param[in]  KEK      - Key Encryption Key, an octet string of length 16 bytes.
   @param[in]  WrapKey  - Wrapped Key, an octet string of length 40 bytes.
   @param[out] Key      - Plain key, an octet string of length 32 bytes.

   @return CE2Error_t - CE2_OK, or error message
                         CE2_CMLA_IS_NOT_SUPPORTED
                         CE2_CMLA_UNWRAP_ILLEGAL_KEK_PTR_ERROR
                         CE2_CMLA_UNWRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR
                         CE2_CMLA_UNWRAP_ILLEGAL_KEY_PTR_ERROR

*/
CE2CIMPORT_C  CE2Error_t CE2_CMLA_Unwrap(
                            CE2_CMLA_KEK_t            KEK,      /*in*/
                            CE2_CMLA_WRAPPED_KEY_t    WrapKey,  /*in*/
                            CE2_CMLA_UNWRAPPED_KEY_t  Key       /*out*/ );


/**************************************************************************
 *	              CE2_CMLA_RSA_Encrypt function                          *
 **************************************************************************/
/**
   @brief  The CE2_CMLA_RSA_Encrypt primitive implements algorithm CMLA_RSA_Encrypt
           [CMLA v1.0-051221].
   @param[in]  PublKey     - A pointer to to structure containing user RSA Public Key.
   @param[in]  Message	   - A pointer to message of length 128 bytes.
   @param[out] EncrMessage - A pointer to output encrypted message of length 128 bytes.

   @return CE2Error_t - CE2_OK, or error message
                         CE2_CMLA_IS_NOT_SUPPORTED
                         CE2_CMLA_RSA_ENCRYPT_ILLEGAL_PUB_KEY_PTR_ERROR
                         CE2_CMLA_RSA_ENCRYPT_ILLEGAL_MESSAGE_PTR_ERROR
                         CE2_CMLA_RSA_ENCRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR
                         CE2_CMLA_RSA_ENCRYPT_MESSAGE_INTEGER_TOO_LONG_ERROR

*/
CE2CIMPORT_C  CE2Error_t  CE2_CMLA_RSA_Encrypt(
					                CE2_RSAUserPubKey_t      *UserPublKey_ptr, /*in*/
					                CE2_CMLA_RSA_MESSAGE_t    Message,         /*in*/
					                CE2_CMLA_RSA_MESSAGE_t    EncrMessage);     /*out*/
					                


/**************************************************************************
 *	              CE2_CMLA_RSA_Decrypt function                          *
 **************************************************************************/
/**
   @brief  The CE2_CMLA_RSA_Decrypt primitive implements the CMLA_RSA_Decrypt algorithm
           [CMLA v1.0-051221].

   @param[in]  PrivKey     - A pointer to to structure containing user RSA Private Key.
   @param[in]  EncrMessage - A pointer to input encrypted message of length 128 bytes.
   @param[out] Message     - A pointer to output decrypted message of length 128 bytes.

   @return CE2Error_t - CE2_OK, or error message
                         CE2_CMLA_IS_NOT_SUPPORTED
                         CE2_CMLA_RSA_ENCRYPT_ILLEGAL_PRIV_KEY_PTR_ERROR
                         CE2_CMLA_RSA_ENCRYPT_ILLEGAL_MESSAGE_PTR_ERROR
                         CE2_CMLA_RSA_ENCRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR
                         CE2_CMLA_RSA_DECRYPT_MESSAGE_INTEGER_TOO_LONG_ERROR

*/
CE2CIMPORT_C  CE2Error_t  CE2_CMLA_RSA_Decrypt(
					                CE2_RSAUserPrivKey_t     *UserPrivKey_ptr, /*in*/
					                CE2_CMLA_RSA_MESSAGE_t    EncrMessage,     /*in*/
					                CE2_CMLA_RSA_MESSAGE_t    Message);        /*out*/



#ifdef __cplusplus
}
#endif

#endif
